"""
配置管理器
"""
import os
import yaml
from typing import Optional, Dict, Any
from pathlib import Path
from loguru import logger


class ConfigManager:
    """配置管理器"""
    
    def __init__(self, config_file: str = "config.yaml"):
        self.config_file = Path(config_file)
        self.config: Dict[str, Any] = {}
        self._load_config()
    
    def _load_config(self):
        """加载配置"""
        if self.config_file.exists():
            try:
                with open(self.config_file, 'r', encoding='utf-8') as f:
                    self.config = yaml.safe_load(f) or {}
                logger.info(f"加载配置文件: {self.config_file}")
            except Exception as e:
                logger.error(f"加载配置文件失败: {e}")
                self.config = {}
        else:
            # 创建默认配置
            self.config = self._default_config()
            self._save_config()
    
    def _default_config(self) -> Dict[str, Any]:
        """默认配置"""
        return {
            'servers': [
                'http://localhost:8080',
            ],
            'client': {
                'id': '',
                'token': '',
                'name': '',
            },
            'tunnels': [],
            'auto_reconnect': True,
            'reconnect_interval': 5,
        }
    
    def _save_config(self):
        """保存配置"""
        try:
            self.config_file.parent.mkdir(parents=True, exist_ok=True)
            with open(self.config_file, 'w', encoding='utf-8') as f:
                yaml.dump(self.config, f, allow_unicode=True, default_flow_style=False)
            logger.info(f"保存配置文件: {self.config_file}")
        except Exception as e:
            logger.error(f"保存配置文件失败: {e}")
    
    def get(self, key: str, default: Any = None) -> Any:
        """获取配置值"""
        keys = key.split('.')
        value = self.config
        for k in keys:
            if isinstance(value, dict):
                value = value.get(k)
                if value is None:
                    return default
            else:
                return default
        return value
    
    def set(self, key: str, value: Any):
        """设置配置值"""
        keys = key.split('.')
        config = self.config
        for k in keys[:-1]:
            if k not in config:
                config[k] = {}
            config = config[k]
        config[keys[-1]] = value
        self._save_config()
    
    def get_server_urls(self) -> List[str]:
        """获取服务器URL列表"""
        servers = self.get('servers', ['http://localhost:8080'])
        if isinstance(servers, str):
            # 兼容旧配置格式
            return [servers]
        return servers if isinstance(servers, list) else ['http://localhost:8080']
    
    def set_server_urls(self, urls: List[str]):
        """设置服务器URL列表"""
        self.set('servers', urls)
    
    def get_client_id(self) -> str:
        """获取客户端ID"""
        return self.get('client.id', '')
    
    def get_client_token(self) -> str:
        """获取客户端Token"""
        return self.get('client.token', '')
    
    def get_client_name(self) -> str:
        """获取客户端名称"""
        return self.get('client.name', '')
    
    def set_client_info(self, client_id: str, token: str, name: str):
        """设置客户端信息"""
        self.set('client.id', client_id)
        self.set('client.token', token)
        self.set('client.name', name)
    
    def get_tunnels(self) -> list:
        """获取隧道列表"""
        return self.get('tunnels', [])
    
    def add_tunnel(self, tunnel_config: Dict[str, Any]):
        """添加隧道配置"""
        tunnels = self.get_tunnels()
        tunnels.append(tunnel_config)
        self.set('tunnels', tunnels)
    
    def remove_tunnel(self, tunnel_id: str):
        """移除隧道配置"""
        tunnels = self.get_tunnels()
        tunnels = [t for t in tunnels if t.get('id') != tunnel_id]
        self.set('tunnels', tunnels)

